//
// (c) 2023 wesolutions GmbH
// All rights reserved.
//

import QtQml 2.12
import QtQuick 2.12

import wesual.Ui 1.0

FocusScope {
    id : uiTextEdit

    signal editingFinished()

    property bool bindable : false
    property alias readOnly : edit.readOnly

    property string text
    property string placeHolderText

    property int horizontalMargin : 0
    property int verticalMargin   : 0

    readonly property bool hover : viewport.containsMouse

    property int minimumDisplayLines : 2
    property int maximumDisplayLines : 10

    signal updateRequested(string newValue)

    function beginEdit(text) {
        edit.beginEdit(text);
    }

    function cancelEdit() {
        edit.cancelEdit();
    }

    implicitWidth : 250
    implicitHeight : {
        const lineHeight = metrics.height;
        const min = lineHeight * minimumDisplayLines + 2 * verticalMargin;
        const max = lineHeight * maximumDisplayLines + 2 * verticalMargin;

        return  Math.max(min, Math.min(max, edit.implicitHeight));
    }

    baselineOffset : edit.baselineOffset

    MouseArea {
        id : viewport

        property real contentY

        function sanitizeContentY() {
            contentY = Math.max(0, Math.min(edit.height - height, contentY));
        }

        function ensureCursorVisible() {
            if (!uiTextEdit.activeFocus)
                return;

            const cursorRect = edit.cursorRectangle;
            if (contentY > cursorRect.y) {
                contentY = cursorRect.y - verticalMargin;
            } else if (contentY + height < cursorRect.y + cursorRect.height) {
                contentY = cursorRect.y
                        + cursorRect.height
                        - height
                        + verticalMargin;
            }
        }

        acceptedButtons : Qt.NoButton
        anchors.fill : uiTextEdit
        clip : true
        cursorShape  : Qt.IBeamCursor
        hoverEnabled : true

        onWheel : {
            if (!uiTextEdit.activeFocus) {
                wheel.accepted = false;

                return;
            }

            contentY -= wheel.angleDelta.y;
            sanitizeContentY();
        }

        onHeightChanged : sanitizeContentY()

        TextEdit {
            id : edit

            function beginEdit(text) {
                forceActiveFocus();
                if (text) {
                    edit.insert(edit.text.length, text);
                }
            }

            function endEdit() {
                commitEdit();
                uiTextEdit.editingFinished();
            }

            function cancelEdit() {
                reset();
                uiTextEdit.focus = false;
            }

            function commitEdit() {
                if (uiTextEdit.bindable) {
                    uiTextEdit.updateRequested(edit.text);
                } else {
                    uiTextEdit.text = edit.text;
                }
                reset();
            }

            function reset() {
                text = Qt.binding(() => uiTextEdit.text);
                cursorPosition = Qt.binding(() => uiTextEdit.text.length);
                viewport.contentY = 0;
            }

            activeFocusOnTab : true
            bottomPadding : uiTextEdit.verticalMargin
            color : {
                if (!uiTextEdit.enabled) {
                    return UiColors.getColor(UiColors.MediumLightGrey27);
                } else if (uiTextEdit.readOnly) {
                    return UiColors.getColor(UiColors.MediumGrey45);
                } else {
                    return UiColors.getColor(UiColors.DarkGrey80);
                }
            }

            font : UiFonts.getFont(UiFonts.Light, 14)
            leftPadding : uiTextEdit.horizontalMargin
            rightPadding : uiTextEdit.horizontalMargin
            selectByMouse : true
            selectedTextColor : UiColors.getColor(UiColors.White)
            selectionColor : UiColors.getColor(UiColors.SelectionGreen)
            text : uiTextEdit.text
            topPadding : uiTextEdit.verticalMargin
            width : viewport.width
            wrapMode : Text.Wrap
            y : -viewport.contentY

            onActiveFocusChanged : {
                if (activeFocus) {
                    viewport.ensureCursorVisible();
                }
            }

            cursorPosition : uiTextEdit.text.length

            onCursorRectangleChanged : viewport.ensureCursorVisible()
            onEditingFinished : endEdit()
            onHeightChanged : viewport.sanitizeContentY()

            Keys.onEscapePressed : cancelEdit()
            Keys.onTabPressed : {
                const next = nextItemInFocusChain(true);
                if (next) {
                    next.forceActiveFocus();
                }
            }
            Keys.onBacktabPressed : {
                const next = nextItemInFocusChain(false);
                if (next) {
                    next.forceActiveFocus();
                }
            }
        }
    }

    UiTextContextMenu {
        textEditor : edit
    }

    FontMetrics {
        id : metrics

        font : edit.font
    }

    // Place holder text
    Text {
        id : label

        anchors.fill : uiTextEdit
        bottomPadding : uiTextEdit.verticalMargin
        color : UiColors.getColor(UiColors.MediumGrey33)
        elide : Text.ElideRight
        font : edit.font
        leftPadding : uiTextEdit.horizontalMargin
        maximumLineCount : uiTextEdit.minimumDisplayLines
        rightPadding : uiTextEdit.horizontalMargin
        text : uiTextEdit.placeHolderText
        topPadding : uiTextEdit.verticalMargin
        visible : !uiTextEdit.activeFocus && uiTextEdit.text.length < 1
        wrapMode : Text.Wrap
    }

    Rectangle {
        id : fadeout

        anchors {
            bottom : uiTextEdit.bottom
            bottomMargin : 1
            left : uiTextEdit.left
            right : uiTextEdit.right
        }
        height : 32
        visible : !uiTextEdit.activeFocus && edit.height > uiTextEdit.height
        z : 2

        gradient : Gradient {
            GradientStop {
                position : 0
                color : "#00FFFFFF"
            }
            GradientStop {
                position : 1
                color : "#FFFFFF"
            }
        }
    }
}
